/**********************************************************************/
/* Name:    blackbar.c                                                */
/* For:     Topfield TF4000 PVR, TF5000 PVR                           */
/* Author:  Christian Ghisler, support@ghisler.com                    */
/* Licence: This TAP is freeware. You may use this code in your own   */
/*          projects free of charge.                                  */
/* Descr.:  This program shows a black bar at the bottom of the       */
/*          screen, which can be used to hide the scrolling band in   */
/*          some news channels (e.g. N24)                             */
/* Usage:   UHF button turns display on and off                       */
/*          UP/DOWN buttons enlarge/reduce the bar size               */
/*          EXIT button quits the TAP                                 */
/**********************************************************************/

#include <tap.h>

int initialize(void)
{
	return 0;
}

#ifdef Topfield4000
static api_table event_save;
#define windownr 0
#else
static int windownr=0;

#define printf TAP_Print

#define ID_BLACKBAR 0x80000001 

TAP_ID( ID_BLACKBAR ); 
TAP_PROGRAM_NAME("Blackbar"); 
TAP_AUTHOR_NAME("GHISLER"); 
TAP_DESCRIPTION("This tap program shows a black bar on screen"); 
TAP_ETCINFO(__DATE__); 

#endif

static int line_visible,start_line;
static int startup_time=0;

/**********************************************************************/
/* Functions                                                          */
/**********************************************************************/


int ShowHideWindow(int show)
{
	if (show) {
		char *my_palette;

		TAP_ExitNormal();      // leaving normal receiver operation

#ifdef Topfield4000
		my_palette = malloc(1024);
		if (!my_palette) return 0;
		TAP_GetDefaultPalette(my_palette);
		TAP_Osd_Delete(windownr);
		TAP_Osd_Create(windownr, 0, 0, 720, 576, 256, 2);
		TAP_Osd_SetLut(2, my_palette, 16);
		TAP_Osd_Ctrl(windownr, 1);
		TAP_Osd_SetTransparency(windownr, 0);
		free(my_palette);
#else
		windownr=TAP_Osd_Create(0, 0, 720, 576, 0, FALSE);
		TAP_Osd_SetTransparency(windownr, 255);
#endif
	} else {
		TAP_Osd_Delete(windownr);
		TAP_EnterNormal();      // re-entering normal receiver operation
	}
	return 1;
}

void UpdateBlackBar(int now_line,int prev_line)
{
	if (line_visible) {
		// Clear first
		if (now_line>prev_line && now_line-prev_line<5)
			TAP_Osd_FillBox(windownr, 0, prev_line-1, 720, now_line-prev_line+1, COLOR_None);
	    TAP_Osd_FillBox(windownr, 0, now_line, 720, 576-now_line, COLOR_Black);
	}
}

/**********************************************************************/
/* Callbacks                                                          */
/**********************************************************************/

static int event_handler(int code)
{
	int old_line;
//	printf("event_handler called, code=%i\n",code);

    switch (code) {
    case RKEY_Uhf:
		code=0;
		line_visible = 1-line_visible;
		ShowHideWindow(line_visible);
		if (line_visible)
			UpdateBlackBar(start_line,start_line);
		break;
	case RKEY_Exit:
		if (line_visible) {
			printf("Program exit\n");
			code=0;
			line_visible=0;
			ShowHideWindow(line_visible);
#ifdef Topfield4000
			__entrytable__[0x30 / 4] = event_save;
            TAP_ClearCallback();
#endif
			TAP_Exit();
		}
		break;
	case RKEY_ChUp:
	case RKEY_ChDown:
		if (line_visible) {
			old_line = start_line;
			if (code==RKEY_ChUp) {
				start_line -= 1;
				if (start_line<100) start_line=100;
			} else {
				start_line += 1;
				if (start_line>560) start_line=560;
			}
			UpdateBlackBar(start_line,old_line);
			code=0;
		}
		break;
	default:                    // on any other key, hide but not terminate
		if (line_visible) {
			line_visible=0;
			ShowHideWindow(line_visible);
		}
	}
	return code;
}

//static int tsr_handler(int code,int param1,int param2)
static int tsr_handler(int code)
{
	int eventdelta,nowtime;

	nowtime=TAP_GetTick();
	// auto-show 1 second after start
	eventdelta=nowtime-startup_time;
	if (eventdelta<0) eventdelta=-eventdelta;
	if (startup_time && eventdelta>=100) {
		startup_time=0;
		if (line_visible==0) {   // user didn't turn it on by himself
			line_visible = 1;
			ShowHideWindow(1);
			UpdateBlackBar(start_line,start_line);
		}
	}
    return code;
}

#ifndef Topfield4000
dword TAP_EventHandler( word event, dword param1, dword param2 )
{
	if( event == EVT_KEY )
		return event_handler(param1);
	if( event == EVT_IDLE )
		return tsr_handler(param1);
	return param1;
}

#endif

/**********************************************************************/
/* TAP_main                                                           */
/**********************************************************************/

#ifdef Topfield4000
int TAP_main(void)
#else
int TAP_Main(void)
#endif
{
	start_line = 500;
	line_visible = 0;

#ifdef Topfield4000
    event_save = __entrytable__[0x30 / 4];
    __entrytable__[0x30 / 4] = event_handler;

    TAP_SetTSRCallback(tsr_handler);
#endif

	startup_time=TAP_GetTick();

	return 1; /* 0 means return, 1 means stay TSR */
}


