/**********************************************************************/
/* Name:    signal.c                                                  */
/* For:     Topfield TF4000 PVR, TF5000 PVR                           */
/* Author:  Christian Ghisler, support@ghisler.com                    */
/* Licence: This TAP is freeware. You may use this code in your own   */
/*          projects free of charge. Use at your own risk!            */
/* Descr.:  This program shows a window with two bars for signal      */
/*          strength and signal quality. Useful for adjusting the     */
/*          satellite dish.                                           */
/* Usage:   UHF button turns tap display on and off                   */
/*          EXIT button quits the TAP                                 */
/**********************************************************************/

#include <tap.h>

#define COLOR_WINDOW COLOR_DarkCyan
#define COLOR_WINDOWTEXT COLOR_White
#define COLOR_BTNFACE COLOR_DarkBlue
#define COLOR_BTNTEXT COLOR_White
#define COLOR_BTNSHADOW COLOR_Black
#define COLOR_BTNHIGHLIGHT COLOR_Blue

#ifndef Topfield4000

#define printf TAP_Print

// needs to be requested from Topfield!
#define ID_SIGNAL 0x80000000

TAP_ID( ID_SIGNAL );
TAP_PROGRAM_NAME("Signal");
TAP_AUTHOR_NAME("GHISLER");
TAP_DESCRIPTION("This tap program shows signal level and quality");
TAP_ETCINFO(__DATE__);

#endif

/*****************************************************************************/
/* Global variables */
/*****************************************************************************/

#ifdef Topfield4000
static api_table event_save;
#define windownr 0
#else
static int windownr;
#endif

static int window_visible=0;
static int last_event=0;
static int startup_time=0;
static TYPE_Window *menu=NULL;
static int last_lvl=-1;
static int last_qual=-1;


/*****************************************************************************/
/* Functions */
/*****************************************************************************/

int initialize(void)
{
	return 0;
}

void DrawBar(window,x,y,w,h,percent,color1,color2)
{
	char percentstr[16];

	// draw two bars, an active and inactive part
	if (percent<100)
		TAP_Osd_FillBox(window, x+w*percent/100, y, w*(100-percent)/100, h, color2);
	if (percent>0)
		TAP_Osd_FillBox(window, x, y, w*percent/100, h, color1);

	sprintf(percentstr,"%i %%",percent);

#ifdef Topfield4000
	TAP_Osd_PutS(window,x, y, x+w, percentstr,COLOR_Black, 0xFF, 0, 2, 0, ALIGN_CENTER);
#else
	TAP_Osd_PutS(window,x, y, x+w, percentstr,COLOR_Black, COLOR_None, 0, 2, 0, ALIGN_CENTER);
#endif
}

void ShowOrHide(int show)
{
	if (show) {
		char *my_palette;

		TAP_ExitNormal();      // leaving normal receiver operation, receiver hides its OSDs

#ifdef Topfield4000
	    my_palette = malloc(1024);
    	if (!my_palette) return;
	    TAP_GetDefaultPalette(my_palette);
    	TAP_Osd_Delete(windownr);  // only that window!
	    TAP_Osd_Create(windownr, 0, 0, 720, 576,256,2);
    	TAP_Osd_SetLut(2, my_palette, 16);
	    TAP_Osd_Ctrl(windownr, 1);
    	free(my_palette);
#else
	    windownr=TAP_Osd_Create(0, 0, 720, 576,0, FALSE );
#endif

	    menu = (TYPE_Window*)malloc(sizeof(TYPE_Window));
    	if (!menu) return;

		memset(menu, 0, sizeof(TYPE_Window));
	    TAP_Win_SetDefaultColor(menu);

		TAP_Osd_FillBox	(windownr, 100, 340, 500, 120, COLOR_WINDOW);

	    TAP_Win_Create( menu, windownr, 100, 340, 500, 120, 0, 0 );
		TAP_Win_Draw(menu);
	    TAP_Win_SetTitle( menu, "Signal indicator", 0, FNT_Size_1622 );
#ifdef Topfield4000
		TAP_Osd_PutString(windownr,120, 390, -1, "Signal strength:",COLOR_WINDOWTEXT, COLOR_WINDOW, 0, 2, 0);
		TAP_Osd_PutString(windownr,120, 445, -1, "Signal quality:",COLOR_WINDOWTEXT, COLOR_WINDOW, 0, 2, 0);
#else
		TAP_Osd_PutString(windownr,120, 390, -1, "Signal strength:",COLOR_WINDOWTEXT, COLOR_None, 0, 2, 0);
		TAP_Osd_PutString(windownr,120, 445, -1, "Signal quality:",COLOR_WINDOWTEXT, COLOR_None, 0, 2, 0);
#endif
		last_lvl=-1;
		last_qual=-1;
	} else {
    	if (!menu) return;
		
	    TAP_Win_Delete(menu);
		free(menu);
		menu=NULL;
    	TAP_Osd_Delete(windownr);
		TAP_EnterNormal();      // re-entering normal receiver operation
	}
}

static void UpdatePercentBars()
{
	int lvl,qual;
	lvl=TAP_GetSignalLevel();
	qual=TAP_GetSignalQuality();

	if (lvl!=last_lvl) {
		DrawBar(windownr,120,420,500,25,lvl,COLOR_Green,COLOR_WINDOW);
		last_lvl=lvl;
	}
	if (qual!=last_qual) {
		DrawBar(windownr,120,475,500,25,qual,COLOR_Yellow,COLOR_WINDOW);
		last_qual=qual;
	}
}

/*****************************************************************************/
/* Callbacks */
/*****************************************************************************/

static int event_handler(int code)
{
	switch (code) {
	case RKEY_Uhf:
		window_visible = 1-window_visible;
		last_event=0;   // immediate display of the bars
		ShowOrHide(window_visible);
		return 0;
	case RKEY_Exit:
		if (window_visible) {
			printf("Program exit\n");
			if (menu) {
#ifdef Topfield4000
				TAP_Osd_PutString(windownr,400, 390, -1, "Program closing...",COLOR_WINDOWTEXT, COLOR_WINDOW, 0, 2, 0);
#else
				TAP_Osd_PutString(windownr,400, 390, -1, "Program closing...",COLOR_WINDOWTEXT, COLOR_None, 0, 2, 0);
#endif
				TAP_Delay(200);
			}

#ifdef Topfield4000
			__entrytable__[0x30 / 4] = event_save;
			TAP_ClearCallback();
#else
			ShowOrHide(0);
#endif
			TAP_Exit();
		}
		break;
	default:
		if (window_visible) {
			window_visible=0;
			ShowOrHide(window_visible);
		}
		break;
	}
	return code;
}

static int tsr_handler(int code)
{
	int eventdelta,nowtime;

	nowtime=TAP_GetTick();
	// auto-show 1 second after start
	eventdelta=nowtime-startup_time;
	if (eventdelta<0) eventdelta=-eventdelta;
	if (startup_time && eventdelta>=100 && window_visible==0) {
		startup_time=0;
		window_visible=1;
		ShowOrHide(window_visible);
		last_event=0;
	}
	// only update every 10th idle event
	eventdelta=nowtime-last_event;
	if (eventdelta<0) eventdelta=-eventdelta;
	if (eventdelta>=10 && window_visible) {
		UpdatePercentBars();
		last_event=nowtime;
	}			
    return code;
}

#ifndef Topfield4000
dword TAP_EventHandler( word event, dword param1, dword param2 )
{
    if( event == EVT_KEY )
		return event_handler(param1);
    if( event == EVT_IDLE )
		return tsr_handler(param1);
	return param1;
}

#endif

/*****************************************************************************/
/* TAP_main */
/*****************************************************************************/

#ifdef Topfield4000
int TAP_main(void)
#else
int TAP_Main(void)
#endif
{
#ifdef Topfield4000
	event_save = __entrytable__[0x30 / 4];
	__entrytable__[0x30 / 4] = event_handler;

	TAP_SetTSRCallback(tsr_handler);
#endif

	startup_time=TAP_GetTick();

	return 1; /* 0 means return, 1 means stay TSR */
}


