#include "topfield.h"
#include <string.h>

HANDLE tfdisk;

DWORD cluster_size = 0x800 << 9; /* Default for disks <= 128 GB */

tf_sector0 root_sector;
unsigned long long lba_sectors;

directory_entry_t directory[1024];
tsdbname fullnames[1024];

DWORD filetable[1024];

DWORD	fat[FATSIZE];

BYTE	cluster_buffer[MAX_CLUSTER_SIZE];

char tf_error[1024];

int tf_read(__off64_t position, LPBYTE buffer, DWORD size, int swapit)
{
	DWORD bytes_read;
	int i;
	WORD *p;

	LARGE_INTEGER pos;
	pos.QuadPart = position;
	
	if (lba_sectors && ((position >> 9) >= lba_sectors)) {
		sprintf(tf_error, "Attempt to read after end of disk!\n");
		return 1;
	}
	if (SetFilePointer(tfdisk, pos.LowPart, &pos.HighPart, FILE_BEGIN) == INVALID_SET_FILE_POINTER) {
		sprintf(tf_error, "SetFilePointer failed on position %I64x with code %d\n", position, GetLastError());
		return 1;
	}
	if (!ReadFile(tfdisk, buffer, size, &bytes_read, NULL)) {
		sprintf(tf_error, "ReadFile failed on position %I64x with code %d\n", position, GetLastError());
		return 1;
	}

	if (bytes_read != size) {
		sprintf(tf_error, "ReadFile did not read %ld bytes but %ld bytes at position %I64x (cluster %I64d)\n", size, bytes_read, position, position/cluster_size);
		return 1;
	}
	if (!swapit) return 0;
	for (i=0, p=(WORD*)buffer; i < size; i += 2, p++) {
		*p = ntohs(*p);
	}
	return 0;
}

int read_fat(void (*cb)(int))
{
	int i;
	DWORD fatitems = (DWORD)((lba_sectors / ntohs(root_sector.cluster_size)) - 1);
	DWORD fatsectors = (fatitems * 3 + 511) >> 9;
	BYTE buffer[fatsectors << 9];
	DWORD fat1[fatitems], fat2[fatitems];

	int fat1bad = 0;
	int fat2bad = 0;
	DWORD j = 0, k = 0, l = 0;
	directory_entry_t *p = (directory_entry_t *)cluster_buffer;
	directory_entry_t nextdir[256];

	DWORD sb;

	/* Topfield file systems allows max 0x300 sectors of FAT data
	   This gives us 256*512 = 131072 FAT entries; Disks up to 128 GB
	   use 1 MB allocation blocks, larger disks use larger allocation
	   blocks (ie. 160 GB disk uses 0xa00 sectors allocation block
	   (1.5 MB)).
	*/
	if (fatitems > FATSIZE) {
		fatitems = FATSIZE;
		fatsectors = 0x300;
	}

	/* Read in the first FAT; reading from physical drives has to be
	   in multiples of whole sectors.
	*/
	if (tf_read(0x100 << 9, buffer, fatsectors << 9, 1)) {
		return 1;
	}

	/* Walk over the FAT data just read */
	for (i = 0; i < fatitems; i++) {
		/* Create fat1 temporrary array */
		fat1[i] = (buffer[3*i] << 16) | (buffer[3*i+1] << 8) | buffer[3*i+2];
		/* Check each item for validity */
		if (fat1[i] < 0xfffffb && fat1[i] > (lba_sectors >> 11)) {
			sprintf(tf_error, "FAT1 %u item %u wrong!\n", i, fat1[i]);
			(*cb)(-1);
			
			fat1[i] = ~0L;
			fat1bad = 1;
		}
	}

	/* Read in the second copy of FAT */
	if (tf_read(0x400 << 9, buffer, fatsectors << 9, 1)) {
		return 1;
	}

	for (i = 0; i < fatitems; i++) {
		fat2[i] = (buffer[3*i] << 16) | (buffer[3*i+1] << 8) | buffer[3*i+2];
		if (fat2[i] < 0xfffffb && fat2[i] > (lba_sectors >> 11)) {
			sprintf(tf_error, "FAT2 %u item %u wrong!\n", i, fat2[i]);
			(*cb)(-1);
			fat2[i] = ~0L;
			fat2bad = 1;
		}
		/* Check if FAT1 and FAT2 are identical; ignore
		   error items marked in phase 1
		*/
		if (	(fat1[i] != (~0L)) && 
			(fat2[i] != (~0L)) && 
			(fat1[i] != fat2[i]) ) {
			sprintf(tf_error, "FAT: %x: %x != %x\n", i, fat1[i], fat2[i]);
			(*cb)(-1);
		}
	}

	/* Copy a valid fat into global variable for later use */
	if (!fat1bad) {
		memcpy(fat, fat1, fatitems * sizeof(DWORD));
	}
	if (fat1bad && !fat2bad) {
		memcpy(fat, fat2, fatitems * sizeof(DWORD));
	}
	if (fat1bad && fat2bad) {
		sprintf(tf_error, "No valid FAT24 found!\n");
		return 1;
	}

	/* Now look for corsslinked or inaccessible blocks */
	memset(fat1, 0, fatitems * sizeof(DWORD));
	memset(fat2, 0, fatitems * sizeof(DWORD));

	fat1bad = 0;

	/* Mark all starts of chains of blocks into fat1 array */
	for (i = 0; i < fatitems; i++) {
		if (fat[i] > 0xfffffb) continue;
		for (j = 0; j < fatitems; j++) {
			if (fat[j] == i) break;
		}
		if (j == fatitems) fat1[k++] = i; /* new start of chain */
	}

	/* Walk over all chains and check for crosslinking using fat2 as tmp */
	for (i = 0; i < k; i++) {
		j = fat1[i]; /* get start of chain */
		while (j != 0xfffffe && j != (~0L)) {
			if (fat2[j]) {
				sprintf(tf_error, "Crosslinked block %x in chain %u and %u\n", j, fat2[j]-1, i);
				(*cb)(-1);
				fat1bad = 1;
			}
			fat2[j] = i+1; /* mark with chain # */
			j = fat[j];
		}
	}

	/* Count free blocks, look for strange blocks */
	/* Is this really necessary? */
	j = 0;
	for (i = 0; i < fatitems; i++) {
		if (fat[i] == 0xffffff) {
			j++;
		} else if (fat[i] != 0xfffffe && fat[i] != (~0L)) {
			if (!fat2[i]) {
				sprintf(tf_error, "FAT item %x bad\n", i);
				(*cb)(-1);
				fat1bad = 1;
			}
		}
	}
//	sprintf(tf_error, "FAT: %u freed blocks\n", j);
//	(*cb)(-1);

	/* Copy fat into fat2 so that we can overwrite it */
	memcpy(fat2, fat, fatitems * sizeof(DWORD));

	/* Read in the __ROOT__ directory */
	if (tf_read((__off64_t)(cluster_size), cluster_buffer, cluster_size, 1)) return -1;
	/* Walk over the the root directory, check file length, save
	   other directories
	*/
	for (; p->type != 0xff; p++) {
		sb=ntohl(p->start_block);
		/* Find which chain this file belongs to */
		for (i = 0; i < k; i++) {
			if (sb == fat1[i]) break;
		}
		if (p->type != 0xf0 && p->type != 0xf1) {
			if (fat2[sb] == 0xffffff) {
				sprintf(tf_error, "Crosslinked single block %x\n", i);
				(*cb)(-1);
				fat1bad = 1;
			}
		}
		fat2[sb] = 0xffffff;
		/* Count the length and mark as referenced */
		for (j = 0; sb != 0xfffffe; j++) {
			fat2[sb] = 0xffffff;
			sb = fat[sb];
		}
		if ((p->type != 0xf2) && (j != ntohl(p->count_of_blocks))) {
			sprintf(tf_error, "File %s has incorrect length %u/%u\n", p->name, ntohl(p->count_of_blocks), j);
			(*cb)(-1);
			fat1bad = 1;
		}

		if (p->type == 0xf2) {
			memcpy(&(nextdir[l++]), p, sizeof(directory_entry_t));
		}
	}


	for (k = 0; k < l; k++) {
		/* Read in the directory */
		if (tf_read((__off64_t)(ntohl(nextdir[k].start_block)+1)*(cluster_size), cluster_buffer, cluster_size, 1)) return -1;
		for (p = (directory_entry_t *)cluster_buffer; p->type != 0xff; p++) {
			sb=ntohl(p->start_block);
			if (p->type != 0xf0 && p->type != 0xf1) {
				if (fat2[sb] == 0xffffff) {
					sprintf(tf_error, "Crosslinked single block %x\n", i);
					(*cb)(-1);
					fat1bad = 1;
				}
			}
			fat2[sb] = 0xffffff;
			for (j = 0; sb != 0xfffffe; j++) {
				fat2[sb] = 0xffffff;
				sb = fat[sb];
			}
			if (p->type != 0xf2 && j != ntohl(p->count_of_blocks)) {
				sprintf(tf_error, "File %s has incorrect length %u/%u\n", p->name, ntohl(p->count_of_blocks), j);
				(*cb)(-1);
				fat1bad = 1;
			}
			if (p->type == 0xf2) {
				memcpy(&(nextdir[l++]), p, sizeof(directory_entry_t));
			}
		}
	}

	/* Check for inaccessible blocks */
	for (i = 0, l = 0; i < fatitems; i++) {
		if (fat2[i] != 0xffffff) {
			sprintf(tf_error, "FAT item %x not accessible\n", i);
			(*cb)(-1);
			fat1bad = 1;
			l++;
		}
	}
	if (fat1bad) sprintf(tf_error, "FAT has errors, please correct!\n");
	return 0; /* we actually ignore non-fatal errors */
}

void convert_date_time(BYTE *p, int *y, int *m, int *d, int *hr, int *min)
{
/* Please see EN 300 468 Annex C - Conversion between time and date conventions */
/* The EN is available from ETSI */

#define	MJD_scale	10000L

	long MJDdate = ((p[0]<<8) | p[1]) * MJD_scale;
	int month, day, year;
	int year_diff, month_diff;

	*hr = p[2];
	*min = p[3];

	if (MJDdate) {
		year = (MJDdate - 150782000L) / 3652500L;
		year_diff = ((year * 3652500L) / MJD_scale) * MJD_scale;
		month = (MJDdate - 149561000L - year_diff) / 306001L;
		month_diff = ((month * 306001L) / MJD_scale) * MJD_scale;
		day = (MJDdate - 149560000L - year_diff - month_diff) / MJD_scale;
		if (month > 13) {
			year++;
			month -= 13;
		} else {
			month--;
		}
		*y = year + 1900;
		*m = month;
		*d = day;
	} else {
		*y = 0; *m = 0; *d = 0;
	}
}


int parse_dir(DWORD cluster, DWORD size)
{
	directory_entry_t *p = (directory_entry_t *)cluster_buffer;
	DWORD	tsdb = -1;
	int i, count, j;
	int entries = 0;

	if (tf_read((__off64_t)(cluster+1)*cluster_size, cluster_buffer, size, 1)) return -1;
	for (; p->type != 0xff; p++) {
		if (p->type == 0xd1) {
			memcpy(&directory[entries], p, sizeof(directory_entry_t));
			entries++;
		}
		if (!strcmp(p->name, "__FILETSDB__.ss")) {
			tsdb = htonl(p->start_block);
		}
	}

	if (tsdb != -1) {
		if (tf_read((__off64_t)(tsdb+1)*cluster_size, cluster_buffer, cluster_size, 1)) return -1;
		count = htonl(*((DWORD*)cluster_buffer));
		for (i = 1; i <= count; i++) {
			for (j = 0; j < entries; j++) {
				if (!strcmp(directory[j].name, ((tsdbname *)cluster_buffer)[i])) {
					filetable[i-1] = j;
					break;
				}
			}
			if (j < entries) {
				strcpy(fullnames[i-1], ((tsdbname *)cluster_buffer)[i]);
			}
		}
	}
	return count;
}

int open_tfdisk(void (*cb)(int))
{
	char pathname[MAX_PATH];
	BYTE sector0[512];
	int code;
	int drive;
	static char signature[] = "TOPFIELD PVR HDD";
	DISK_GEOMETRY geometry;
	DWORD length;
	DWORD empty, last, n_empty;
	int inrange;
	int found = 0;

	tfdisk = CreateFile("tfdisk.img", GENERIC_READ, 0, (LPSECURITY_ATTRIBUTES)NULL, OPEN_EXISTING, 0, (HANDLE)0);
	if (tfdisk != INVALID_HANDLE_VALUE) {
		LARGE_INTEGER pos;
		pos.QuadPart = 0;
		length = SetFilePointer(tfdisk, pos.LowPart, &pos.HighPart, FILE_END);
		if (length == INVALID_SET_FILE_POINTER) {
			sprintf(tf_error, "Cannot determine size of saved image.\n");
			return 1;
		}
		pos.LowPart = length;
		lba_sectors = pos.QuadPart / 512LL;
		if (tf_read(0, (PBYTE)(&root_sector), sizeof(root_sector), 1)) return 1;
		if (strncmp(root_sector.signature, signature, sizeof(signature))) {
			sprintf(tf_error, "Bad signature in saved image!\n");
			return 1;
		}
		cluster_size = ntohs(root_sector.cluster_size) << 9;
		if (read_fat(cb))
			return 1;
		return 0;
	}

	for (drive = 0; drive < 8; drive++) {
		sprintf(pathname, "\\\\.\\\\PHYSICALDRIVE%d", drive);
		tfdisk = CreateFile(pathname, GENERIC_READ, 0, (LPSECURITY_ATTRIBUTES)NULL, OPEN_EXISTING, 0, (HANDLE)0);
		if (tfdisk != INVALID_HANDLE_VALUE) {
			DeviceIoControl(tfdisk, IOCTL_DISK_GET_DRIVE_GEOMETRY,
				NULL, 0,
				&geometry, sizeof(geometry), &length, NULL);
			if (geometry.MediaType == 12) {
				lba_sectors = geometry.Cylinders.QuadPart * geometry.TracksPerCylinder * geometry.SectorsPerTrack;
				if (tf_read(0, (PBYTE)(&root_sector), sizeof(root_sector), 1)) return 1;
				if (!strncmp(root_sector.signature, signature, sizeof(signature))) {
					found = 1;
					cluster_size = ntohs(root_sector.cluster_size) << 9;
					break;
				}
			}
		}
		close(tfdisk);
	}

	if (!found) {
		sprintf(tf_error, "No Topfield disk found!\n");
		return 1;
	}

	if (read_fat(cb))
		return 1;
	return 0;
}

int dump_file(DWORD file, void (*cb)(int))
{
	DWORD cluster = ntohl(directory[file].start_block);
	DWORD max_cluster = ntohl(directory[file].count_of_blocks);
	DWORD size = cluster_size;
	FILE *fp;
	int percent = -1;
	long long clusters = 0;
	char newname[512];

	make_name(newname, directory[file].name, "");
	fp = fopen(newname, "wb");
	if (fp == NULL) {
		sprintf(tf_error, "Cannot open '%s' for output\n", directory[file].name);
		return 1;
	}

	do {
		if (tf_read((__off64_t)(cluster+1)*cluster_size, cluster_buffer, cluster_size, 1)) return 1;
		cluster = fat[cluster];
		if (cluster == 0xfffffe) {
			size = cluster_size - ntohl(directory[file].empty_in_last_block);
		}
		if (fwrite(cluster_buffer, 1, size, fp) < size) {
			sprintf(tf_error, "Write failed (disk full?)\n");
			fclose(fp);
			return 1;
		}
		clusters++;
		if (percent != (int)(((clusters * 100LL) / max_cluster))) {
			percent = (int)((clusters * 100LL) / max_cluster);
			(*cb)(percent);
		}
	} while (cluster != 0xfffffe);
	fclose(fp);
	return 0;
}


buf_rec_file_t *open_recording_file(DWORD file, void (*cb)(int))
{
	DWORD cluster = ntohl(directory[file].start_block);
	buf_rec_file_t *p = malloc(sizeof(buf_rec_file_t));
	int i;

	if (!p) {
		sprintf(tf_error, "malloc for bufferred reading failed\n");
		return 0;
	}
	p->file = file;
	p->cb = cb;
	if (tf_read((__off64_t)(cluster+1)*cluster_size, p->buffer, cluster_size, 1)) return 0;
	p->cluster = fat[cluster];

	memcpy((void *)&(p->header), p->buffer, sizeof(recording_header_t));

	for (i = sizeof(recording_header_t); i < sizeof(recording_header_t)+sizeof(bookmarks_t)+sizeof(ts_packet); i++) {
		if (p->buffer[i] == TS_SYNC && p->buffer[i+sizeof(ts_packet)] == TS_SYNC) {
			break;
		}
	}
	if (i==sizeof(recording_header_t)+sizeof(bookmarks_t)+sizeof(ts_packet)) {
		sprintf(tf_error, "Start of TS packets not found in file %s.\n", directory[file].name);
		free(p);
		return 0;
	}
	p->ptr = i; /* where packets start */
	return p;
}

void close_recording_file(buf_rec_file_t *p)
{
	if (!p) return;
	free(p);
}
	
ts_packet *read_ts_packet(buf_rec_file_t *p)
{
	ts_packet *result;
	static char local_ts_packet[sizeof(ts_packet)];
	DWORD limit;

again:
	limit = (p->cluster == 0xfffffe) ? cluster_size - ntohl(directory[p->file].empty_in_last_block) : cluster_size;
	if (!limit) return 0; /* EOF */
	result = (ts_packet *)(p->buffer + p->ptr);
	if (p->ptr == limit) {
		p->ptr = sizeof(ts_packet);
		if (p->cluster != 0xfffffe) {
			if (tf_read((__off64_t)(p->cluster+1)*cluster_size, p->buffer, cluster_size, 1)) return 0;
			p->cluster = fat[p->cluster];
			if (p->buffer[0] != TS_SYNC) {
				p->ptr = 1;
				goto again;
			}
			return (ts_packet *)(p->buffer);
		}
		return 0; /* end of file */
	}
	if (p->ptr + sizeof(ts_packet) <= limit) {
		if (p->buffer[p->ptr] != TS_SYNC) {
			p->ptr++;
			goto again;
		}
		p->ptr += sizeof(ts_packet);
		return result;
	}
	if (p->buffer[p->ptr] != TS_SYNC) {
		p->ptr++;
		goto again;
	}
	memcpy(local_ts_packet, (void*)result, limit-p->ptr);
	if (p->cluster != 0xfffffe) {
		if (tf_read((__off64_t)(p->cluster+1)*cluster_size, p->buffer, cluster_size, 1)) return 0;
		p->cluster = fat[p->cluster];
		p->ptr = sizeof(ts_packet) - (limit - p->ptr);
		memcpy(local_ts_packet + sizeof(ts_packet)-p->ptr, p->buffer, p->ptr);
		return (ts_packet*)(&local_ts_packet);
	}
	return 0; /* end of file */
}

int copy_disk(void (*cb)(int))
{
	FILE *fp;
	long long cluster = 0;
	long long max_cluster;

	int percent = -1;
	int tmp;

	max_cluster = (DWORD)(lba_sectors >> 11);
	fp = fopen("tfdisk.img", "wb");
	if (fp == NULL) {
		sprintf(tf_error, "Cannot open output\n");
		return 1;
	}
	do {
		if (tf_read((__off64_t)(cluster)*cluster_size, cluster_buffer, cluster_size, 0)) {
			sprintf(tf_error, "Error while reading TF disk\n");
			fclose(fp);
			return 1;
		}
		if (fwrite(cluster_buffer, 1, cluster_size, fp) < cluster_size) {
			sprintf(tf_error, "Write failed (disk full?)\n");
			fclose(fp);
			return 1;
		}
		cluster++;
		tmp = (cluster * 100) / max_cluster;
		if (percent != tmp) {
			percent = tmp;
			(*cb)(percent);
		}
	} while (cluster < max_cluster);
	fclose(fp);
	return 0;
}

void make_name(char *newname, const char *oldname, const char *extension)
{
	strcpy(newname, oldname);
	strcat(newname, extension);
	while (*newname) {
		if (!strchr("0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ.-+=;$!@#$%^&()[]{}", *newname))
			*newname = ' ';
		newname++;
	}
}
