#pragma pack(1)

#ifdef	linux
#define	_LARGEFILE64_SOURCE
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#else
#include <windows.h>
#include <largeint.h>
#include <winioctl.h>
#endif

#include <stdio.h>
#include <stdlib.h>

#include "byteorder.h" /* Use my own */

/* Cluster size is variable since disks > 128GB were introduced */
/* This size is for disks up to around 220 GB */
#define	MAX_CLUSTER_SIZE (0xe00*0x200)
#define	FATSIZE	(256*512)
/* This was too small in previous versions and has caused problems with
   many channels who use long PES packets which were truncated without warning.
*/
#define	BUFFER_SIZE	0x100000
#define	MAX_PID	0x2000

#ifdef	linux
#ifndef	MAX_PATH
#define	MAX_PATH	512
#endif
typedef	int	HANDLE;
typedef	unsigned char BYTE, *PBYTE, *LPBYTE;
typedef unsigned short WORD, *PWORD;
typedef unsigned long DWORD, *PDWORD;
typedef __off64_t LARGE_INTEGER;
#else
typedef long long __off64_t;
#endif


/*** Topfield and DVB data types ***/

typedef struct {
	DWORD	magic;
	char	signature[28]; /* "TOPFIELD PVR HDD" */
	WORD	version;
	WORD	cluster_size;
	DWORD	x;
	DWORD	first_empty;
	DWORD	empty_in_root;
	DWORD	fat_crc32;
	BYTE	filler[512-52];
} tf_sector0;

typedef	BYTE	tsdbname[512];


/* The following types are derived from the DVB standards and should not depend
   on how Topfield implements them. They have been derived directly from the
   relevant ETSI documentation. The field names are mostly verbatim from the
   standards texts.
*/

#define	TS_SYNC	0x47

typedef struct {
	BYTE sync_byte;
//	DWORD transport_error_indicator:1;
//	DWORD payload_unit_start_indicator:1;
//	DWORD transport_priority_indicator:1;
	WORD pid;
	BYTE continuity_counter:4;
	BYTE adaptation_field_control:2;
	BYTE transport_scrambling_control:2;
	BYTE data[184];
} ts_packet;

/* Topfield internal structures, grossly incomplete and volatile to change.
   Do not use these unless you actually must. You have been warned.
*/

typedef struct {
	BYTE	sat;
	BYTE	polarity;
	WORD	frequency;
	WORD	symbolrate; /* in kS/s */
	WORD	network_id;
} transponder_description_t;

typedef struct {
	WORD	service_number_and_flags; /* bit 26 - radio/TV */
	WORD	event_id;
	DWORD	start_time;
	DWORD	end_time;
	WORD	duration;
	BYTE	running_status;
	BYTE	length_of_name;
	char		name_of_event_and_description[130];
	WORD	service_ID;
	DWORD	field_94;
} event_data_t;	/* sizeof = 0x98 = 152 */

typedef struct {
	DWORD	end_date_time_1;
	DWORD	end_date_time_2;
	WORD	duration; /* in minutes */
	WORD	service_number;
	WORD	service_flags; /* also event ID? */
	BYTE	tuner;
	BYTE	field_F;
	WORD	transponder_index_and_flags;
	WORD	svc_ID;
	WORD	pmt_PID;
	WORD	pcr_PID;
	WORD	video_PID;
	WORD	audio_PID;
	char		service_name[24];
	transponder_description_t	transponder_description;
	event_data_t	event_data;
} recording_header_t; /* length = 0xd4 = 212 */

typedef struct {
	BYTE	type;	/* 0 */
	BYTE	data[7]; /* 1 */
	DWORD	start_block; /* 8 */
	DWORD	count_of_blocks; /* C */
	DWORD	empty_in_last_block; /* 10 */
	char		name[108]; /* 14 */
} directory_entry_t; /* sizeof = 0x80 = 128 */

typedef	DWORD bookmarks_t[64];


typedef struct {
	int	ptr;
	DWORD	cluster;
	DWORD	file;
	void (*cb)(int);
	recording_header_t	header;
	BYTE	buffer[MAX_CLUSTER_SIZE];
} buf_rec_file_t;

typedef struct demux {
	int counter;
	int packet_ptr;
	int repack_size;
	long long total_size;
	unsigned long long pcr, pcr_mark;
	void *arg;
	int (*func)(struct demux *d);
	int (*repack_func)(struct demux *d);
	BYTE buffer[BUFFER_SIZE];
} demux_t;

typedef demux_t *demux_table_t[MAX_PID];

/*** GLOBAL VARIABLES ***/

/* Operating system handle to the file representing the physical disk or
   saved disk image.
*/
extern HANDLE tfdisk;

extern DWORD cluster_size;

/* Topfield file system superblock as found in physical sector 0 on disk */
extern tf_sector0 root_sector;

/* Total number of sectors on disk, current ATA standard allows up to 48bit
   sector counts.
*/
extern unsigned long long lba_sectors;

/* The root directory of the Topfield disk, recording files only (type D1) */
extern directory_entry_t directory[1024];

/* File order translation table to show files in the same order as on the STB */
extern DWORD filetable[1024];

/* Table of full file names (512 characters maximum including termination '\0'
*/
extern tsdbname fullnames[1024];

/* FAT24 converted to regular DWORD in PC byte order */
extern DWORD	fat[FATSIZE];

/* for PES output, maximum size of output PES packet, 0 if no repacketing
   is required. Does not work at the moment!!!
*/
extern repack;

/* String with information message (for callback) or fatal error (error return).
*/
extern char tf_error[1024];

/*** Top-level functions used in tfget ***/

/* Search for a valid Topfield TF4000PVR disk in the following
   order:
	1. file "tfdisk.img" in current directory
	2. physical disk connected to one of the IDE ports and
	   recognized by the operating system (no partitions!)
   If the disk is found, the superblock is read as well as the FAT
   table into RAM. The variable lba_sectors is set to a total number
   of the sectors on the disk.
   If the disk was found and is valid, return 0, otherwise return non-0.
   In case of error, the global variable tf_error contains a string
   describing the error.
*/
int open_tfdisk(void (*cb)(int));

/* Read in the root directory (called __ROOT__), only recording files
   (file type 0xD1). Both short (104 chars maximum) and long (511 chars maximum)
   names are read. The order of files is the same as on the Topfield recording
   list display (the order is stored in a the file containing the long names).
   returns the number of files found, or <= 0 if error.
   The cluster parameter is the cluster number of the root directory.
   The size parameter is the size of the root directory as specified in the
    superblock.
*/
int parse_dir(DWORD cluster, DWORD size);

/* Copy whole disk to a PC file
   The cb parameter is a callback function, which is called
   with percent complete integer parameter; if the parameter is negative,
   the tf_error variable contains an information message (not a fatal error).
   Returns 0 if OK, non-0 if error. The tf_error variable contains
   description of error in non-0 is returned.
*/
int copy_disk(void (*cb)(int));

/* Copy one file in raw (as-is) format to a PC file
   The file parameter is an index in the directory global variable
   The cb parameter is a callback function, which is called
   with percent complete integer parameter; if the parameter is negative,
   the tf_error variable contains an information message (not a fatal error).
   Returns 0 if OK, non-0 if error. The tf_error variable contains
   description of error in non-0 is returned.
*/
int dump_file(DWORD file, void (*cb)(int));

/* Copy one file by extracting the default audio track only in ES format.
   The file parameter is an index in the directory global variable
   The cb parameter is a callback function, which is called
   with percent complete integer parameter; if the parameter is negative,
   the tf_error variable contains an information message (not a fatal error).
   Returns 0 if OK, non-0 if error. The tf_error variable contains
   description of error in non-0 is returned.
*/
int convert_mp3(DWORD file, void (*cb)(int));

/* Copy one file by extracting the video track only in ES format.
   The file parameter is an index in the directory global variable
   The cb parameter is a callback function, which is called
   with percent complete integer parameter; if the parameter is negative,
   the tf_error variable contains an information message (not a fatal error).
   Returns 0 if OK, non-0 if error. The tf_error variable contains
   description of error in non-0 is returned.
*/
int convert_m2v(DWORD file, void (*cb)(int));

/* Copy one file by extracting the video and default audio tracks only
   in PES format.
   The file parameter is an index in the directory global variable
   The cb parameter is a callback function, which is called
   with percent complete integer parameter; if the parameter is negative,
   the tf_error variable contains an information message (not a fatal error).
   Returns 0 if OK, non-0 if error. The tf_error variable contains
   description of error in non-0 is returned.
*/
int convert_m2p(DWORD file, void (*cb)(int));
int convert_vob(DWORD file, void (*cb)(int));

/* Copy one file by droping the Topfield proprietary header
   and copying only PMT, video and default audio packets.
   There is an attempt to reconstruct the PAT (not recorded in the data),
   but the CRC32 is not valid yet.
   The file parameter is an index in the directory global variable
   The cb parameter is a callback function, which is called
   with percent complete integer parameter; if the parameter is negative,
   the tf_error variable contains an information message (not a fatal error).
   Returns 0 if OK, non-0 if error. The tf_error variable contains
   description of error in non-0 is returned.
*/
int convert_ts(DWORD file, void (*cb)(int));

/* functions for handling recoding files on the level
   of TS packets.
*/

buf_rec_file_t *open_recording_file(DWORD file, void (*cb)(int));
void close_recording_file(buf_rec_file_t *p);
ts_packet *read_ts_packet(buf_rec_file_t *p);

/* functions for demultiplexing/filtering the recording files */

demux_table_t *new_demux_table(void);
int demux_set(demux_table_t *t, int pid, int (*func)(demux_t *), void *arg, int repack_size, int (*repack_func)(demux_t *));
void free_demux_table(demux_table_t *t);
void demux_loop(demux_table_t *t, buf_rec_file_t *p);

/* Output functions for ES and PES packets */
int out_es(demux_t *d);
int out_pes(demux_t *d);

/*** Low-level functions ***/

/* Read data from Topfield disk opened by open_tfdisk.
   position is a 64bit offset on the disk (must be even!)
   buffer is a pointer to where the data will be stored
   size is the number of bytes to be read (must be even!)
   swapit should be set to 1 to swap bytes, 0 to return data as-is
	(the PowerPC processor writes data to the IDE drive with bytes
	swapped due to the BigEndian format used by PowerPC - as opposed
	to LittleEndian used by the x86 processors; as-is data are
	used only for disk image copy, otherwise swapit should be always
	set to 1)
   Returns 0 if OK, non-0 if error. The tf_error variable contains
   description of error in non-0 is returned.
*/
int tf_read(__off64_t position, LPBYTE buffer, DWORD size, int swapit);

/* Read in the FAT24 table into a global variable. There are 2 FAT24 tables
   on every TF4000PVR disk, the position of both is fixed (they start at
   sectors 0x100 and 0x400). There are
   3 bytes (24 bits) in each FAT entry (compare with FAT12, FAT16 and FAT32
   in various versions of MS-DOS/Windows).
   The function validates both FAT tables and tries to use the one
   with no errors (this is consitent with what the Topfield software does).
   Returns 0 if OK, non-0 if error. The tf_error variable contains
   description of error in non-0 is returned.
*/
int read_fat(void (*cb)(int));

/* Convert date and time from DVB format to YMD/HM
   Used for file creation date and time
*/
void convert_date_time(BYTE *p, int *y, int *m, int *d, int *hr, int *min);

/* Create an output file name from the Topfield file name */
void make_name(char *newname, const char *oldname, const char *extension);
