#include <string.h>
#include "topfield.h"

demux_table_t *new_demux_table(void)
{
	demux_table_t *t;
	int pid;

	t = malloc(sizeof(demux_table_t));
	if (!t) {
		sprintf(tf_error, "Allocation of demux table failed!\n");
		return NULL;
	}
	for (pid = 0; pid < MAX_PID; pid++)
		(*t)[pid] = NULL;
	return t;
}


int demux_set(demux_table_t *t, int pid, int (*func)(demux_t *), void *arg, int repack_size, int (*repack_func)(demux_t *))
{
	demux_t *p;

	p = malloc(sizeof(demux_t));
	if (!p) {
		sprintf(tf_error, "malloc of demux buffer failed!\n");
		return 1;
	}
	p->counter = -1;
	p->packet_ptr = 0;
	p->total_size = 0;
	p->func = func;
	p->repack_func = repack_func;
	p->arg = arg;
	p->repack_size = repack_size;
	p->pcr = ~0;
	p->pcr_mark = ~0;
	(*t)[pid] = p;
	return 0;
}

void free_demux_table(demux_table_t *t)
{
	int pid;
	for (pid = 0; pid < MAX_PID; pid++)
		if ((*t)[pid])
			free((*t)[pid]);
	free(t);
}

void demux_loop(demux_table_t *t, buf_rec_file_t *p)
{
	ts_packet *tsp;

	long long	total = 0,
		processed = 0,
		bad_sync = 0,
		outsequence = 0,
		scrambled = 0;
	long long expected = ((long long)(cluster_size) * ntohl(directory[p->file].count_of_blocks)) / sizeof(ts_packet);

	demux_t *d;

	int ptr;

	int percent = -1;
	int tmp;

	unsigned long long prev_pcr = 0;
	long long pcr_mark = 0;
	double rate = 0.0;
	unsigned long long pcr = 0;
	unsigned long long last_pcr = 0;

	while((tsp = read_ts_packet(p))) {
		total++;
		tmp = (100 * total) / expected;
		if (percent != tmp) {
			percent = tmp;
			(*(p->cb))(percent);
		}
		if (tsp->sync_byte != 0x47) {
			bad_sync++;
			continue;
		}
		d = (*t)[ntohs(tsp->pid) & 0x1fff];
		ptr = (tsp->adaptation_field_control & 2)? 1 + tsp->data[0] : 0;
		if (ptr > 1 && (tsp->data[1] & 0x10)) { /* PCR flag */
			pcr = (((unsigned long long)(tsp->data[2]) << 25) +
				((unsigned long long)(tsp->data[3]) << 17) +
				(tsp->data[4] << 9) +
				(tsp->data[5] << 1) +
				((tsp->data[6] >> 7) & 1))*300 + 
				((tsp->data[6] << 8) & 0x100) +
				tsp->data[7];
			if (pcr_mark) rate = ((total-pcr_mark)*188.0*8.0)/(pcr-prev_pcr);
//			printf("PID %x PCR %I64u delta=%I64d bytes=%I64d rate=%f\n", ntohs(tsp->pid) & 0x1fff, pcr, pcr-prev_pcr, (total - pcr_mark)*188, rate*27.0e6);
			prev_pcr = pcr;
			pcr_mark = total;
		}
		if (!d) continue;

		if (tsp->transport_scrambling_control) {
			scrambled++;
			d->packet_ptr = 0; /* ignore this PES packet */
			continue;
		}
		if (d->counter >= 0 && (tsp->continuity_counter) != d->counter) {
				outsequence++;
				d->packet_ptr = 0; /* ignore this PES packet */
				d->counter = ((tsp->continuity_counter) + 1) & 0xf;
				continue;
		}
		processed++;
		d->counter = ((tsp->continuity_counter) + 1) & 0xf;
		if ((tsp->adaptation_field_control & 1) == 0) continue; /* no data */
		pcr = prev_pcr + (unsigned long long)((total - pcr_mark)*188*8 / (1.05*rate));
		if (pcr < last_pcr) {
			pcr = last_pcr;
		//	printf("delayed ");
		} else {
			last_pcr = pcr;
		}
//		printf("calculated pcr is %I64d\n", pcr);
		d->pcr = pcr;
		if (ntohs(tsp->pid)&0x4000) { /* payload start */
			if (d->packet_ptr) {
				if ((*(d->func))(d))
					break;
			}
			
			memcpy(d->buffer, tsp->data+ptr, 184-ptr);
			d->packet_ptr = 184-ptr;
			d->total_size = d->packet_ptr;
		} else if (d->packet_ptr) {
			if (d->packet_ptr + 184 - ptr <= BUFFER_SIZE) {
				d->total_size += 184 - ptr;
				if (d->repack_size && d->packet_ptr + 184 - ptr > d->repack_size) {
					int len_to_fit = d->repack_size - d->packet_ptr;
					memcpy(d->buffer+d->packet_ptr, tsp->data+ptr, len_to_fit);
					d->packet_ptr = d->repack_size;
	
					if ((*(d->repack_func))(d))
						break;
					memcpy(d->buffer+d->packet_ptr, tsp->data+ptr+len_to_fit, 184-ptr-len_to_fit);
					d->packet_ptr += 184 - ptr - len_to_fit;
				} else {
					memcpy(d->buffer+d->packet_ptr, tsp->data+ptr, 184-ptr);
					d->packet_ptr += 184-ptr;
				}
			} else {
				sprintf(tf_error, "\nPacket too long!\n");
				(*(p->cb))(-1);
			}
		}
	}

	if (scrambled) {
		sprintf(tf_error, "%I64d packets scrambled\n", scrambled);
		(*(p->cb))(-1);
	}
	if (outsequence) {
		sprintf(tf_error, "%I64d packets out of sequence\n", outsequence);
		(*(p->cb))(-1);
	}
	if (bad_sync) {
		sprintf(tf_error, "%I64d packets with bad sync\n", bad_sync);
		(*(p->cb))(-1);
	}
}

int get_vinfo(demux_t *d, long *bitrate)
{
	BYTE *b = d->buffer;
	int found = 0;
	int c = 0;
	int br;

	*bitrate = 0;

	while (found < 4 && c + 4 < d->packet_ptr) {
		if (b[c+0] == 0x00 && b[c+1] == 0x00 && b[c+2] == 0x01 && b[c+3] == 0xb3)
			found = 4;
		else
			c++;
	}

	if (!found) return 0;
	c += 4;
	if (c+12 >= d->packet_ptr) return 0;

	fprintf(stderr, "Video: size = %d x %d", ((b[c+1] & 0xf0) >> 4) | b[c] << 4, ((b[c+1] & 0x0f) << 8 ) | b[c+2]);

	switch ((b[c+3]&0xf0)>>4) {
	case 1:
		fprintf(stderr, ", aspect 1:1");
		break;
	case 2:
		fprintf(stderr, ", aspect 4:3");
		break;
	case 3:
		fprintf(stderr, ", aspect 16:9");
		break;
	case 4:
		fprintf(stderr, ", aspect 2.21:1");
		break;
	case 0:
	case 5 ... 15:
		fprintf(stderr, ", aspect RESERVED");
		break;
	}

	switch (b[c+3]&0x0f) {
	case 1:
		fprintf(stderr, ", frame rate 23.976");
		break;
	case 2:
		fprintf(stderr, ", frame rate 24");
		break;
	case 3:
		fprintf(stderr, ", frame rate 25");
		break;
	case 4:
		fprintf(stderr, ", frame rate 29.97");
		break;
	case 5:
		fprintf(stderr, ", frame rate 30");
		break;
	case 6:
		fprintf(stderr, ", frame rate 50");
		break;
	case 7:
		fprintf(stderr, ", frame rate 60");
		break;
	case 0:
        case 8 ... 15:
		fprintf(stderr, ", frame rate RESERVED");
		break;
	}

	br = 400 * (((b[c+4] << 10) & 0x0003fc00UL) |
		((b[c+5]<<2) & 0x000003fcUL) |
		(((b[c+6] & 0xc0) >> 6) & 0x3UL));

	fprintf(stderr, ", bit rate = %.2f Mbit/s\n", br/1000000.);
	*bitrate = br;
	return 1;
}

int get_ainfo(demux_t *d, long *bitrate)
{
	BYTE *b = d->buffer;
	int found = 0;
	int c = 9 + b[8];
	int br;

	static const unsigned int bitrates[3][16] =
		{{0,32,64,96,128,160,192,224,256,288,320,352,384,416,448,0},
		 {0,32,48,56,64,80,96,112,128,160,192,224,256,320,384,0},
		 {0,32,40,48,56,64,80,96,112,128,160,192,224,256,320,0}};

	static const int freq[4] = {441, 480, 320, 0};

	*bitrate = 0;

	while (found < 2 && c + 2 < d->packet_ptr) {
		if ((b[c+0] == 0xff) && ((b[c+1] & 0xf8) == 0xf8))
			found = 2;
		else
			c++;
	}

	if (!found) return 0;
	if (c+3 >= d->packet_ptr) return 0;

	fprintf(stderr, "Audio: Layer %d", 4-((b[c+1]&6)>>1));

	br = bitrates[3-((b[c+1]&6)>>1)][b[c+2]>>4];

	fprintf(stderr, ", bit rate=%d kb/s", br);

	fprintf(stderr, ", frequency=%2.1f kHz\n", freq[(b[c+2]&0x0c)>>2]/10.);

	*bitrate = 1000*br;

	return 2;
}

DWORD trans_pts_dts(BYTE *p)
{
	DWORD w_dts;
	BYTE *av_dts = (BYTE*)(&w_dts);

	av_dts[0] = ((p[0] & 0x06) << 4) |
		((p[1] & 0xfc) >> 2);
	av_dts[1] = ((p[1] & 0x03) << 6) |
		((p[2] & 0xfc) >> 2);
	av_dts[2] = ((p[2] & 0x02) << 6) |
		((p[3] & 0xfe) >> 1);
	av_dts[3] = ((p[3] & 0x01) << 7) |
		((p[4] & 0xfe) >> 1);
	return w_dts;
}

int out_es(demux_t *d)
{
	static int init = 0;
	static long ar = 0;

	int es_ptr = 9 + d->buffer[8];

	if (!init && (d->buffer[3] & 0xe0) == 0xc0)
		init = get_ainfo(d, &ar);

	if (fwrite(d->buffer+es_ptr, 1, d->packet_ptr - es_ptr, (FILE *)(d->arg)) < d->packet_ptr - es_ptr) {
		sprintf(tf_error, "Write to disk failed, disk full?\n");
		return 1;
	}
	return 0;
}
	
int write_ps_header(BYTE *buf, 
		    DWORD   SCR, 
		    long  muxr,
		    BYTE    audio_bound,
		    BYTE    fixed,
		    BYTE    CSPS,
		    BYTE    audio_lock,
		    BYTE    video_lock,
		    BYTE    video_bound,
		    BYTE    stream1,
		    BYTE    buffer1_scale,
		    DWORD   buffer1_size,
		    BYTE    stream2,
		    BYTE    buffer2_scale,
		    DWORD   buffer2_size)                    
{
#if 0
	ps_packet p;
	uint8_t *pts;
	long lpts;
	init_ps(&p);
	
	lpts = htonl(SCR);
	pts = (BYTE *) &lpts;

	
// SCR = PTS
	p.scr[0] = 0x44 | ((pts[0] >> 3)&0x18) | ((pts[0] >> 4)&0x03);
	p.scr[1] = 0x00 | ((pts[0] << 4)&0xF0) | ((pts[1] >> 4)&0x0F);
	p.scr[2] = 0x04 | ((pts[1] << 4)&0xF0) | ((pts[2] >> 4)&0x08)
		| ((pts[2] >> 5)&0x03);
	p.scr[3] = 0x00 | ((pts[2] << 3)&0xF8) | ((pts[3] >> 5)&0x07);
	p.scr[4] = 0x04 | ((pts[3] << 3)&0xF8);
	p.scr[5] = 0x01;
	
	p.mux_rate[0] = (BYTE)(muxr >> 14);
	p.mux_rate[1] = (BYTE)(0xff & (muxr >> 6));
	p.mux_rate[2] = (BYTE)(0x03 | ((muxr & 0x3f) << 2));

	p.stuff_length = 0xF8;
	
	if (stream1 && stream2){
		p.sheader_llength[0] = 0x00;
		p.sheader_llength[1] = 0x0c;

		setl_ps(&p);
		
		p.rate_bound[0] = (BYTE)(0x80 | (muxr >>15));
		p.rate_bound[1] = (BYTE)(0xff & (muxr >> 7));
		p.rate_bound[2] = (BYTE)(0x01 | ((muxr & 0x7f)<<1));

	
		p.audio_bound = (BYTE)((audio_bound << 2)|(fixed << 1)|CSPS);
		p.video_bound = (BYTE)((audio_lock << 7)|
				     (video_lock << 6)|0x20|video_bound);
		p.reserved = (BYTE)(0xFF >> 1);
		
		p.data[0] =  stream2;
		p.data[1] =  (BYTE) (0xc0 | (buffer2_scale << 5) | 
				   (buffer2_size >> 8));
		p.data[2] =  (BYTE) (buffer2_size & 0xff);
		p.data[3] =  stream1;
		p.data[4] =  (BYTE) (0xc0 | (buffer1_scale << 5) | 
				   (buffer1_size >> 8));
		p.data[5] =  (BYTE) (buffer1_size & 0xff);
		
		cwrite_ps(buf, &p, PS_HEADER_L2);
		kill_ps(&p);
		return PS_HEADER_L2;
	} else {
		cwrite_ps(buf, &p, PS_HEADER_L1);
		kill_ps(&p);
		return PS_HEADER_L1;
	}
#else
	return 0;
#endif
}

int repack_pes(demux_t *d)
{
	BYTE *b = d->buffer;
	int result;


	b[4] = (d->packet_ptr - 6) >> 8;
	b[5] = (d->packet_ptr - 6) & 0xff;
	
	result = (*(d->func))(d);
	b[0] = 0;
	b[1] = 0;
	b[2] = 1;
	// keep b[3] as before
	b[4] = 0;
	b[5] = 0;
	b[6] &= 0xf3;
	b[7] = 0;
	b[8] = 0;
	d->packet_ptr = 9;
	return result;
}

int out_pack(demux_t *d)
{
	BYTE *b = d->buffer;
	int pes_length = (b[4]<<8) + b[5];
	static init = 0;
	static long ar = 0, vr = 0, muxr = 0;
	static header = 0;
	int i;

	static DWORD pack_number = 0;

	static BYTE pack_header[] = {
		0x00, 0x00, 0x01, 0xba, /* pack start code */
		0x44, 0x00, 0x04, 0x04, 0x94, 0xab,
		0x01, 0x89, 0xc3,
		0xf8,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	};

#define	pack_h_size	14

	static BYTE padding[] = {
		0x00, 0x00, 0x01, 0xbe, /* padding stream */
		0x00, 0x00,		/* length */
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
		0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	};

		

	DWORD padlen = 0x800 - pack_h_size - d->packet_ptr;

	if (b[0] != 0 || b[1] != 0 || b[2] != 1) {
		sprintf(tf_error, "Bad PES packet start\n");
		return 0;
	}
	if (!pes_length) {
		b[4] = (d->packet_ptr - 6) >> 8;
		b[5] = (d->packet_ptr - 6) & 0xff;
	} else if (pes_length != d->packet_ptr - 6) {
		sprintf(tf_error, "bad PES packet length %d != %d\n", pes_length, d->packet_ptr - 6);
	}

	if (!(init & 1) && (b[3] & 0xf0)==0xe0) {
		init |= get_vinfo(d, &vr);
	}

	if (!(init & 2) && (b[3] & 0xe0)==0xc0) {
		init |= get_ainfo(d, &ar);
	}

	if (init == 3 && !muxr) muxr = (ar + vr)/400;

	if (init == 3) {
		int check;
		BYTE pbuf[1024];
		unsigned long long scr_base = d->pcr / 300;
		int scr_ext = d->pcr % 300;

#if 0
		check = write_ps_header(pbuf,
			scr, 
			muxr, 
			1, /* audio_bound */
			0, /* fixed */
			0, /* CSPS */
			1, /* audio_lock */
			1, /* video_lock */
			1, /* video_bound */
			0xC0, /* stream1 */
			0, 	/* buffer1_scale */
			64, 	/* buffer1_size */
			0xE0, 	/* stream2 */
			1, 	/* buffer2_scale */
			460);	/* buffer2_size */
#endif

#if 0
/* Not yet working */
		if ((b[3] & 0xe0) == 0xe0) { /* its video, is it GOP? */
			int c;
			for (c = 6; c < d->packet_ptr-4; c++) {
				if (b[c+0] == 0x00 && b[c+1] == 0x00 && b[c+2] == 0x01 && b[c+3] == 0xb3) {
					static BYTE navigation_pack[] = {
	0x00, 0x00, 0x01, 0xba, 0x44, 0x00, 0x04, 0x00, // 000
	0x04, 0x01, 0x01, 0x89, 0xc3, 0xf8, 0x00, 0x00, // 008
	0x01, 0xbb, 0x00, 0x12, 0x80, 0xc4, 0xe1, 0x00, // 010
	0xe1, 0x7f, 0xb9, 0xe0, 0xe8, 0xb8, 0xc0, 0x20, // 018
	0xbd, 0xe0, 0x3a, 0xbf, 0xe0, 0x02, 0x00, 0x00, // 020
	0x01, 0xbf, 0x03, 0xd4, 0x00, 0x00, 0x00, 0x00,	// 028
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 030
	0x00, 0x00, 0x00, 0x52, 0x8c, 0x00, 0x00, 0xfb, // 038
	0x4c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 040
	0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	// 048
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 050
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	// 058
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 060
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 068
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 070
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 078
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 080
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 088
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 090
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 098
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0A0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0A8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0b0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0b8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0c0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0c8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0d0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0d8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0e0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0e8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0f0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0f8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 100
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 108
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 110
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 118
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 120
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 128
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 130
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 138
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 140
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 148
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 150
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	// 158
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 160
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 168
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 170
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 178
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 180
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 188
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 190
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 198
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1A0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1A8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1b0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1b8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1c0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1c8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1d0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1d8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1e0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1e8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1f0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 1f8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 200
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 208
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 210
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 218
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 220
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 228
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 230
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 238
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 240
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 248
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 250
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	// 258
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 260
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 268
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 270
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 278
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 280
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 288
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 290
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 298
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2A0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2A8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2b0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2b8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2c0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2c8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2d0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2d8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2e0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2e8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2f0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 2f8

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 300
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 308
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 310
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 318
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 320
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 328
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 330
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 338
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 340
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 348
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 350
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	// 358
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 360
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 368
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 370
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 378
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 380
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 388
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 390
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 398
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3A0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3A8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3b0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3b8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3c0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3c8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3d0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3d8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3e0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3e8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3f0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 3f8

	0x00, 0x00, 0x01, 0xbf, 0x03, 0xfa, 0x01, 0x00, // 400
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 408
	0x00, 0x00, 0xe1, 0x00, 0x00, 0x00, 0x14, 0x00, // 410
	0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x5c, 0x00, // 418
	0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x40, 0x00, // 420
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 428
	0x00, 0x00, 0x00, 0x00, 0x00, 0x52, 0x8c, 0x10, // 430
	0xb2, 0x2b, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, // 438
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 440
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 448
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 450
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 458
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 460
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 468
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 470
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 478
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 480
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 488
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 490
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 498
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 4a0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 4a8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 4b0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 4b8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 4c0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 4c8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 4d0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 4d8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 4e0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 4e8

	0x00, 0x80, 0x00, 0x00, 0xe2, 0xc0, 0x00, 0xd7, // 4f0
	0x56, 0xc0, 0x00, 0x6b, 0xab, 0xc0, 0x00, 0x35, // 4f8
	0xdd, 0xc0, 0x00, 0x11, 0xef, 0x80, 0x00, 0x0d, // 500
	0x74, 0x80, 0x00, 0x0c, 0x8e, 0x80, 0x00, 0x0b, // 508
	0xa8, 0x80, 0x00, 0x0a, 0xc1, 0x80, 0x00, 0x09, // 510
	0xdc, 0x80, 0x00, 0x08, 0xf7, 0x80, 0x00, 0x08, // 518
	0x11, 0x80, 0x00, 0x07, 0x2b, 0x80, 0x00, 0x06, // 520
	0x42, 0x80, 0x00, 0x05, 0x5d, 0x80, 0x00, 0x04, // 528
	0x78, 0x80, 0x00, 0x03, 0x92, 0x80, 0x00, 0x02, // 530
	0xad, 0x80, 0x00, 0x01, 0xc7, 0x80, 0x00, 0x00, // 538
	0xe2, 0x80, 0x00, 0x00, 0xe2, 0x3f, 0xff, 0xff, // 540
	0xff, 0x3f, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, // 548
	0xff, 0x3f, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, // 550
	0xff, 0x3f, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, // 558
	0xff, 0x3f, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, // 560
	0xff, 0x3f, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, // 568
	0xff, 0x3f, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, // 570
	0xff, 0x3f, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, // 578
	0xff, 0x3f, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, // 580
	0xff, 0x3f, 0xff, 0xff, 0xff, 0x3f, 0xff, 0xff, // 588
	0xff, 0x3f, 0xff, 0xff, 0xff, 0xbf, 0xff, 0xff, // 590
	0xff, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, // 598
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5a0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5a8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5b0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5b8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5c0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5c8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5d0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5d8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5e0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5e8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5f0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 5f8

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 600
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 608
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 610
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 618
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 620
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 628
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 630
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 638
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 640
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 648
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 650
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	// 658
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 660
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 668
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 670
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 678
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 680
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 688
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 690
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 698
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6A0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6A8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6b0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6b8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6c0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6c8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6d0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6d8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6e0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6e8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6f0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 6f8

	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 700
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 708
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 710
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 718
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 720
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 728
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 730
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 738
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 740
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 748
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 750
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,	// 758
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 760
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 768
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 770
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 778
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 780
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 788
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 790
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 798
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7A0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7A8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7b0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7b8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7c0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7c8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7d0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7d8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7e0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7e8
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7f0
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 7f8
	};

					/* it's time to write a navigation pack */
					fwrite(navigation_pack, 1, 0x800, (FILE *)(d->arg));
				}
			}
		}
#endif
			
		pack_header[4] = 0x44 +
			((scr_base >> (30 - 3)) & 0x38) + /* 32..30 */
			((scr_base >> (28 - 0)) & 0x03); /* 29..28 */
		pack_header[5] =
			((scr_base >> (20 - 0)) & 0xff); /* 27..20 */
		pack_header[6] = 0x04 +
			((scr_base >> (15 - 3)) & 0xf8) + /* 19..15 */
			((scr_base >> (13 - 0)) & 0x03); /* 14..13 */
		pack_header[7] =
			((scr_base >> (05 - 0)) & 0xff); /* 12..05 */
		pack_header[8] = 0x04 +
			((scr_base << 3) & 0xf8) + /* 04..00 */
			((scr_ext >> (07 - 0)) & 0x03); /* 08..07 */
		pack_header[9] = 0x01 +
			((scr_ext << 1) & 0xfe); /* 06..00 */
		if (padlen < 8) {
			pack_header[13] = 0xf8 + padlen;
			if (fwrite(pack_header, 1, pack_h_size + padlen, (FILE *)(d->arg)) < pack_h_size + padlen) {
				sprintf(tf_error, "Write to disk failed, disk full?\n");
				return 1;
			}
		} else {
			pack_header[13] = 0xf8;
			if (fwrite(pack_header, 1, pack_h_size, (FILE *)(d->arg)) < pack_h_size) {
				sprintf(tf_error, "Write to disk failed, disk full?\n");
				return 1;
			}
		}

		b[4] = (d->packet_ptr - 6) >> 8;
		b[5] = (d->packet_ptr - 6) & 0xff;
		b[3] &= 0xe0; /* Always write as audio/video 0 */
		if (fwrite(b, 1, d->packet_ptr, (FILE*)(d->arg)) < d->packet_ptr) {
			sprintf(tf_error, "Write to disk failed, disk full?\n");
			return 1;
		}
		if (padlen >= 8) {
			padding[4] = (padlen - 6) >> 8;
			padding[5] = (padlen - 6) & 0xff;
			if (fwrite(padding, 1, padlen, (FILE *)(d->arg)) < padlen) {
				sprintf(tf_error, "Write to disk failed, disk full?\n");
				return 1;
			}
		}
		pack_number++;
	}
	return 0;
}

int out_pes(demux_t *d)
{
	BYTE *b = d->buffer;
	int pes_length = (b[4]<<8) + b[5];
	static init = 0;
	static long ar = 0, vr = 0, muxr = 0;
	static header = 0;
	int i;

	if (b[0] != 0 || b[1] != 0 || b[2] != 1) {
		sprintf(tf_error, "Bad PES packet start\n");
		return 0;
	}
	if (!pes_length) {
//		b[4] = (d->packet_ptr - 6) >> 8;
//		b[5] = (d->packet_ptr - 6) & 0xff;
	} else if (pes_length != d->packet_ptr - 6) {
		sprintf(tf_error, "bad PES packet length %d != %d\n", pes_length, d->packet_ptr - 6);
	}

	if (!(init & 1) && (b[3] & 0xf0)==0xe0) {
		init |= get_vinfo(d, &vr);
	}

	if (!(init & 2) && (b[3] & 0xe0)==0xc0) {
		init |= get_ainfo(d, &ar);
	}

	if (init == 3 && !muxr) muxr = (ar + vr)/400;

	if (init == 3 && !header) {
		int check;
		BYTE pbuf[1024];
		int scr = trans_pts_dts(d->buffer) - 3600;

		check = write_ps_header(pbuf,
			scr, 
			muxr, 1, 0, 0, 1, 1, 1,
			0xC0, 0, 64, 0xE0, 1, 460);
		header = 1;
	}

	if (header) {
		if (fwrite(b, 1, d->packet_ptr, (FILE *)(d->arg)) < d->packet_ptr) {
			sprintf(tf_error, "Write to disk failed, disk full?\n");
			return 1;
		}
	}
	return 0;
}

int convert_mp3(DWORD file, void (*cb)(int))
{
	FILE *dest;
	buf_rec_file_t	*p;
	demux_table_t *t;
	char mp3name[256];

	int pid;

	p = open_recording_file(file, cb);
	if (!p) {
		sprintf(tf_error, "Open of recording %s failed.\n", directory[file].name);
		return 1;
	}

	pid = ntohs(p->header.audio_PID) & 0x1fff;
	sprintf(tf_error, "Audio PID is %d\n", pid);
	(*cb)(-1);
	
	make_name(mp3name, directory[file].name, ".mp3");
	dest = fopen(mp3name, "wb");
	if (dest == NULL) {
		close_recording_file(p);
		sprintf(tf_error, "Output file creation failed\n");
		return 2;
	}
	t = new_demux_table();
	if (!t) {
		close_recording_file(p);
		fclose(dest);
		return 1;
	}
	if (demux_set(t, pid, out_es, dest, 0, NULL)) {
		close_recording_file(p);
		fclose(dest);
		free_demux_table(t);
		sprintf(tf_error, "Allocation of audio demux failed!\n");
		return 3;
	}

	demux_loop(t, p);

	free_demux_table(t);

	fclose(dest);
	close_recording_file(p);
	return 0;
}

int convert_m2v(DWORD file, void (*cb)(int))
{
	FILE *dest;
	buf_rec_file_t	*p;
	demux_table_t *t;
	char m2vname[256];

	int pid;

	p = open_recording_file(file, cb);
	if (!p) {
		sprintf(tf_error, "Open of recording %s failed.\n", directory[file].name);
		return 1;
	}

	pid = ntohs(p->header.video_PID) & 0x1fff;
	sprintf(tf_error, "Video PID is %d\n", pid);
	(*cb)(-1);
	
	make_name(m2vname, directory[file].name, ".m2v");
	dest = fopen(m2vname, "wb");
	if (dest == NULL) {
		close_recording_file(p);
		sprintf(tf_error, "Output file creation failed\n");
		return 2;
	}
	t = new_demux_table();
	if (!t) {
		close_recording_file(p);
		fclose(dest);
		return 1;
	}
	if (demux_set(t, pid, out_es, dest, 0, NULL)) {
		close_recording_file(p);
		fclose(dest);
		free_demux_table(t);
		sprintf(tf_error, "Allocation of video demux failed!\n");
		return 3;
	}

	demux_loop(t, p);

	free_demux_table(t);

	fclose(dest);
	close_recording_file(p);
	return 0;
}

int convert_vob(DWORD file, void (*cb)(int))
{
	FILE *dest;
	buf_rec_file_t	*p;
	demux_table_t *t;
	char m2pname[256];

	int vpid, apid;

	p = open_recording_file(file, cb);
	if (!p) {
		sprintf(tf_error, "Open of recording %s failed.\n", directory[file].name);
		return 1;
	}

	vpid = ntohs(p->header.video_PID) & 0x1fff;
	apid = ntohs(p->header.audio_PID) & 0x1fff;
	sprintf(tf_error, "Video PID is %d, audio PID is %d\n", vpid, apid);
	(*cb)(-1);
	
	make_name(m2pname, directory[file].name, ".vob");
	dest = fopen(m2pname, "wb");
	if (dest == NULL) {
		close_recording_file(p);
		sprintf(tf_error, "Output file creation failed\n");
		return 2;
	}

	t = new_demux_table();
	if (!t) {
		close_recording_file(p);
		fclose(dest);
		return 1;
	}
	if (demux_set(t, vpid, out_pack, dest, 2034, repack_pes)) {
		close_recording_file(p);
		fclose(dest);
		free_demux_table(t);
		sprintf(tf_error, "Allocation of video demux failed!\n");
		return 3;
	}
	if (demux_set(t, apid, out_pack, dest, 2034, repack_pes)) {
		close_recording_file(p);
		fclose(dest);
		free_demux_table(t);
		sprintf(tf_error, "Allocation of audio demux failed!\n");
		return 3;
	}

	demux_loop(t, p);

	free_demux_table(t);
	fclose(dest);
	close_recording_file(p);
	return 0;
}
int convert_m2p(DWORD file, void (*cb)(int))
{
	FILE *dest;
	buf_rec_file_t	*p;
	demux_table_t *t;
	char m2pname[256];

	int vpid, apid;

	p = open_recording_file(file, cb);
	if (!p) {
		sprintf(tf_error, "Open of recording %s failed.\n", directory[file].name);
		return 1;
	}

	vpid = ntohs(p->header.video_PID) & 0x1fff;
	apid = ntohs(p->header.audio_PID) & 0x1fff;
	sprintf(tf_error, "Video PID is %d, audio PID is %d\n", vpid, apid);
	(*cb)(-1);
	
	make_name(m2pname, directory[file].name, ".m2p");
	dest = fopen(m2pname, "wb");
	if (dest == NULL) {
		close_recording_file(p);
		sprintf(tf_error, "Output file creation failed\n");
		return 2;
	}

	t = new_demux_table();
	if (!t) {
		close_recording_file(p);
		fclose(dest);
		return 1;
	}
	if (demux_set(t, vpid, out_pes, dest, repack, repack_pes)) {
		close_recording_file(p);
		fclose(dest);
		free_demux_table(t);
		sprintf(tf_error, "Allocation of video demux failed!\n");
		return 3;
	}
	if (demux_set(t, apid, out_pes, dest, repack, repack_pes)) {
		close_recording_file(p);
		fclose(dest);
		free_demux_table(t);
		sprintf(tf_error, "Allocation of audio demux failed!\n");
		return 3;
	}

	demux_loop(t, p);

	free_demux_table(t);
	fclose(dest);
	close_recording_file(p);
	return 0;
}

int convert_ts(DWORD file, void (*cb)(int))
{
	FILE *dest;
	buf_rec_file_t	*p;
	demux_table_t *t;
	char tsname[256];

	int vpid, apid, pmtpid, pid;

	ts_packet *tsp;
	long long total = 0, processed = 0, bad_sync = 0;

	long long expected;

	int percent = -1;
	int tmp;

	static BYTE pat[sizeof(ts_packet)] = {
		0x47,
		0x40, 0x00, /* PID, start */
		0x20, /* payload */
		0x00, /* table ID */
		0x80, 0x0D, /* syntax, length */
		0x00, 0x00, /* TSID */
		0x01, /* version, current/next */
		0x00, /* section number */
		0x00, /* last section number */
		0x00, 0x00, /* program number - offset = 12 */
		0x00, 0x00, /* PMT pid - offset = 14 */
		0x00, 0x00, 0x00, 0x00, /* CRC32 - offset = 16 */
	};

	p = open_recording_file(file, cb);
	if (!p) {
		sprintf(tf_error, "Open of recording %s failed.\n", directory[file].name);
		return 1;
	}

	expected = ((long long)(cluster_size) * ntohl(directory[p->file].count_of_blocks)) / sizeof(ts_packet);

	vpid = ntohs(p->header.video_PID) & 0x1fff;
	apid = ntohs(p->header.audio_PID) & 0x1fff;
	pmtpid = ntohs(p->header.pmt_PID) & 0x1fff;
	sprintf(tf_error, "PMT PID is %d, video PID is %d, audio PID is %d\n", pmtpid, vpid, apid);
	(*cb)(-1);
	
	make_name(tsname, directory[file].name, ".ts");
	dest = fopen(tsname, "wb");
	if (dest == NULL) {
		close_recording_file(p);
		sprintf(tf_error, "Output file creation failed\n");
		return 2;
	}

	pat[12] = p->header.svc_ID >> 8;
	pat[13] = p->header.svc_ID & 0xff;
	pat[14] = p->header.pmt_PID >> 8;
	pat[15] = p->header.pmt_PID & 0xff;

	//fwrite(pat, 1, sizeof(ts_packet), dest);

	while((tsp = read_ts_packet(p))) {
		total++;
		tmp = (100 * total) / expected;
		if (percent != tmp) {
			percent = tmp;
			(*cb)(percent);
		}
		if (tsp->sync_byte != 0x47) {
			bad_sync++;
			continue;
		}

		pid = ntohs(tsp->pid) & 0x1fff;

		if (pid == pmtpid || pid == vpid || pid == apid) {
			processed++;
			if (pid == pmtpid) {
				fwrite(pat, 1, sizeof(ts_packet), dest);
			}
			fwrite(tsp, 1, sizeof(ts_packet), dest);
		}
	}

	fclose(dest);
	close_recording_file(p);
	return 0;
}
