/* tf_disk.c
   routines to read Topfield TF4000PVR disks in PC under Windows NT, 2000, XP

   Copyright (c) Petr Novak  <topfield@centrum.cz>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "topfield.h"
#include <string.h>

#define tf_read_cluster(cno) tf_read((__off64_t)(cno+1)*tf->cluster_size, cluster_buffer, tf->cluster_size, 1)

tf_t tf_instance;
tf_t *tf = &tf_instance;

BYTE	cluster_buffer[MAX_CLUSTER_SIZE];

char tf_error[1024];

static int check_dir(DWORD cluster, DWORD maxdir, DWORD *fat2, DWORD *dirstack, DWORD *numdirs)
{
	directory_entry_t *p = (directory_entry_t *)cluster_buffer;
	DWORD dir_item;

	DWORD sb;
	int j;
	int fat1bad = 0;

	/* read the directory into cluster_buffer */
	if (tf_read_cluster(cluster)) return -1;

	/* Walk over the the root directory, check file length, save
	   other directories
	*/
	for (dir_item = 0; dir_item < maxdir; p++, dir_item++) {
		if (p->type == 0xff) continue;
		sb=ntohl(p->start_block);
		if (p->type != 0xf0 && p->type != 0xf1) {
			if (fat2[sb] == 0xffffff) {
				sprintf(tf_error, "Crosslinked single block %x\n", sb);
				(tf->cb)(-1);
				fat1bad = 2;
			}
		}
		fat2[sb] = 0xffffff;
		/* Count the length and mark as referenced */
		for (j = 0; sb != 0xfffffe; j++) {
			fat2[sb] = 0xffffff;
			sb = tf->fat[sb];
		}
		if ((p->type != 0xf2) && (j != ntohl(p->count_of_blocks))) {
			sprintf(tf_error, "File %s has incorrect length %u/%u\n", p->name, ntohl(p->count_of_blocks), j);
			(tf->cb)(-1);
			fat1bad = 1;
		}

		if (p->type == 0xf2) {
			dirstack[(*numdirs)++] = ntohl(p->start_block);
		}
	}
	return fat1bad;
}

int read_fat(void)
{
	int i;
	DWORD fat1[FATSIZE], fat2[FATSIZE];

	int fat1bad = 0;
	int fat2bad = 0;
	DWORD j = 0, k = 0, l = 0;
	directory_entry_t *p = (directory_entry_t *)cluster_buffer;

	DWORD sb;

	BYTE *buffer;

	int dir_item, maxdir = tf->cluster_size / sizeof(directory_entry_t);

	DWORD numdirs = 0;
	DWORD dirstack[maxdir];

	/* Read in the cluster with both FATs */
	if (tf_read_cluster(-1)) return 1;

	buffer = cluster_buffer + (0x100 << 9);

	/* Walk over the FAT data just read */
	for (i = 0; i < tf->fatitems; i++) {
		/* Create fat1 temporrary array */
		fat1[i] = (buffer[3*i] << 16) | (buffer[3*i+1] << 8) | buffer[3*i+2];
		/* Check each item for validity */
		if (fat1[i] < 0xfffffb && fat1[i] > (tf->lba_sectors >> 11)) {
			sprintf(tf_error, "FAT1 %u item %u wrong!\n", i, fat1[i]);
			(tf->cb)(-1);
			
			fat1[i] = ~0L;
			fat1bad = 1;
		}
	}

	buffer = cluster_buffer + (0x400 << 9);

	for (i = 0; i < tf->fatitems; i++) {
		fat2[i] = (buffer[3*i] << 16) | (buffer[3*i+1] << 8) | buffer[3*i+2];
		if (fat2[i] < 0xfffffb && fat2[i] > (tf->lba_sectors >> 11)) {
			sprintf(tf_error, "FAT2 %u item %u wrong!\n", i, fat2[i]);
			(tf->cb)(-1);
			fat2[i] = ~0L;
			fat2bad = 1;
		}
		/* Check if FAT1 and FAT2 are identical; ignore
		   error items marked in phase 1
		*/
		if (	(fat1[i] != (~0L)) && 
			(fat2[i] != (~0L)) && 
			(fat1[i] != fat2[i]) ) {
			sprintf(tf_error, "FAT: %x: %x != %x\n", i, fat1[i], fat2[i]);
			(tf->cb)(-1);
		}
	}

	/* Copy a valid fat into global variable for later use */
	if (!fat2bad) {
		memcpy(tf->fat, fat2, tf->fatitems * sizeof(DWORD));
	}
	if (fat2bad && !fat1bad) {
		memcpy(tf->fat, fat1, tf->fatitems * sizeof(DWORD));
	}
	if (fat1bad && fat2bad) {
		sprintf(tf_error, "No valid FAT24 found!\n");
		return 1;
	}

	/* Now look for corsslinked or inaccessible blocks */
	memset(fat1, 0, tf->fatitems * sizeof(DWORD));
	memset(fat2, 0, tf->fatitems * sizeof(DWORD));

	fat1bad = 0;

	/* Mark all starts of chains of blocks into fat1 array */
	for (i = 0; i < tf->fatitems; i++) {
		if (tf->fat[i] > 0xfffffb) continue;
		for (j = 0; j < tf->fatitems; j++) {
			if (tf->fat[j] == i) break;
		}
		if (j == tf->fatitems) fat1[k++] = i; /* new start of chain */
	}

	/* Walk over all chains and check for crosslinking using fat2 as tmp */
	for (i = 0; i < k; i++) {
		j = fat1[i]; /* get start of chain */
		while (j <= 0xfffffb && j != (~0L)) {
			if (fat2[j]) {
				sprintf(tf_error, "Crosslinked block %x in chain %u and %u\n", j, fat2[j]-1, i);
				(tf->cb)(-1);
				fat1bad = 1;
			}
			fat2[j] = i+1; /* mark with chain # */
			j = tf->fat[j];
		}
	}

	/* Copy fat into fat2 so that we can overwrite it */
	memcpy(fat2, tf->fat, tf->fatitems * sizeof(DWORD));

	/* Check the __ROOT__ directory */
	fat1bad |= check_dir(0, (((tf->cluster_size - ntohl(tf->superblock.empty_in_root)) & 0xffe00) + 512)/sizeof(directory_entry_t), fat2, dirstack, &numdirs);

	/* Check all other directories */
	for (k = 0; k < numdirs; k++) {
		fat1bad |= check_dir(dirstack[k], tf->cluster_size / sizeof(directory_entry_t), fat2, dirstack, &numdirs);
	}

	/* Check for inaccessible blocks */
	for (i = 0, l = 0; i < tf->fatitems; i++) {
		if (fat2[i] != 0xffffff) {
			sprintf(tf_error, "FAT item %x not accessible\n", i);
			(tf->cb)(-1);
			fat1bad = 1;
			l++;
		}
	}
	if (l) {
		sprintf(tf_error, "Total of %u lost clusters found\n", l);
		(tf->cb)(-1);
	}
	if (fat1bad) sprintf(tf_error, "FAT has errors, please correct!\n");
	if (fat1bad & 2) return 1;
	return 0; /* we actually ignore non-fatal errors */
}

void convert_date_time(BYTE *p, int *y, int *m, int *d, int *hr, int *min)
{
/* Please see EN 300 468 Annex C - Conversion between time and date conventions */
/* The EN is freely available from www.ETSI.ch */

#define	MJD_scale	10000L

	long MJDdate = ((p[0]<<8) | p[1]) * MJD_scale;
	int month, day, year;
	int year_diff, month_diff;

	*hr = p[2];
	*min = p[3];

	if (MJDdate) {
		year = (MJDdate - 150782000L) / 3652500L;
		year_diff = ((year * 3652500L) / MJD_scale) * MJD_scale;
		month = (MJDdate - 149561000L - year_diff) / 306001L;
		month_diff = ((month * 306001L) / MJD_scale) * MJD_scale;
		day = (MJDdate - 149560000L - year_diff - month_diff) / MJD_scale;
		if (month > 13) {
			year++;
			month -= 13;
		} else {
			month--;
		}
		*y = year + 1900;
		*m = month;
		*d = day;
	} else {
		*y = 0; *m = 0; *d = 0;
	}
}


int parse_dir(DWORD cluster, DWORD maxdir, BYTE mask, BYTE value)
{
	directory_entry_t *p = (directory_entry_t *)cluster_buffer;
	DWORD	tsdb = -1;
	int i, count, j;
	int entries = 0;
	tf_dir_t *tempdir;
	DWORD dir_item;

	tempdir = malloc(sizeof(tf_dir_t)*MAX_DIR);
	if (tempdir == NULL) {
		sprintf(tf_error, "Not enough memory to process directory.\n");
		return -1;
	}
	if (tf_read_cluster(cluster)) return -1;

	for (dir_item = 0; dir_item < maxdir; p++, dir_item++) {
		if (p->type == 0xff) continue; /* Ignore deleted files */
		if ((p->type & mask) == value) {
			tf_dir_t *d = tempdir+entries;
			d->type = p->type;
			strcpy(d->name, p->name);
			d->start_block = ntohl(p->start_block);
			d->count_of_blocks = ntohl(p->count_of_blocks);
			d->empty_in_last_block = ntohl(p->empty_in_last_block);
			memcpy(d->data, p->data, sizeof(p->data));
			entries++;
		}
		if (!strcmp(p->name, "__FILETSDB__.ss")) {
			tsdb = htonl(p->start_block);
		}
	}

	if (tsdb != -1) {
		if (tf_read_cluster(tsdb)) return -1;
		count = htonl(*((DWORD*)cluster_buffer));
		for (i = 1; i <= count; i++) {
			for (j = 0; j < entries; j++) {
				if (!strncmp(tempdir[j].name, ((tsdbname *)cluster_buffer)[i], 107)) {
					memcpy(tf->dir+i-1,  tempdir+j, sizeof(tf_dir_t));
					strcpy(tf->dir[j].name, ((tsdbname*)cluster_buffer)[i]);
					break;
				}
			}
		}
		free(tempdir);
		return count;
	} else {
		memcpy(tf->dir, tempdir, entries * sizeof(tf_dir_t));
		free(tempdir);
		return entries;
	}
}

int open_tfdisk(void (*cb)(int))
{
	char pathname[MAX_PATH];
	BYTE sector0[512];
	int code;
	int drive;
	static char signature[] = "TOPFIELD PVR HDD";
	DISK_GEOMETRY geometry;
	DWORD length;
	DWORD empty, last, n_empty;
	int inrange;
	int found = 0;

	tf->cb = cb;
	tf->handle = CreateFile("tfdisk.img", GENERIC_READ, 0, (LPSECURITY_ATTRIBUTES)NULL, OPEN_EXISTING, 0, (HANDLE)0);
	if (tf->handle == INVALID_HANDLE_VALUE)
		tf->handle = CreateFile("..\\tfdisk.img", GENERIC_READ, 0, (LPSECURITY_ATTRIBUTES)NULL, OPEN_EXISTING, 0, (HANDLE)0);
	if (tf->handle != INVALID_HANDLE_VALUE) {
		LARGE_INTEGER pos;
		pos.QuadPart = 0;
		length = SetFilePointer(tf->handle, pos.LowPart, &pos.HighPart, FILE_END);
		if (length == INVALID_SET_FILE_POINTER) {
			sprintf(tf_error, "Cannot determine size of saved image.\n");
			return 1;
		}
		pos.LowPart = length;
		tf->lba_sectors = pos.QuadPart / 512LL;

		if (tf_read(0, (PBYTE)(&(tf->superblock)), sizeof(tf_superblock_t), 1)) return 1;
		if (strncmp(tf->superblock.signature, signature, sizeof(signature))) {
			sprintf(tf_error, "Bad signature in saved image!\n");
			return 1;
		}
		found = 1;
	} else
	for (drive = 0; drive < 8; drive++) {
		sprintf(pathname, "\\\\.\\\\PHYSICALDRIVE%d", drive);
		tf->handle = CreateFile(pathname, GENERIC_READ, 0, (LPSECURITY_ATTRIBUTES)NULL, OPEN_EXISTING, 0, (HANDLE)0);
		if (tf->handle != INVALID_HANDLE_VALUE) {
			DeviceIoControl(tf->handle, IOCTL_DISK_GET_DRIVE_GEOMETRY,
				NULL, 0,
				&geometry, sizeof(geometry), &length, NULL);
			if (geometry.MediaType == 12) {
				tf->lba_sectors = geometry.Cylinders.QuadPart * geometry.TracksPerCylinder * geometry.SectorsPerTrack;
				if (!tf_read(0, (PBYTE)(&(tf->superblock)), sizeof(tf_superblock_t), 1) &&
				    !strcmp(tf->superblock.signature, signature)) {
					found = 1;
					break;
				}
			}
		}
		close(tf->handle);
	}

	if (!found) {
		sprintf(tf_error, "No Topfield disk found!\n");
		return 1;
	}

	tf->cluster_size = ntohs(tf->superblock.cluster_size) << 9;

	/* Topfield file system allows max 0x300 sectors of FAT data
	   This gives us 256*512 = 131072 FAT entries; Disks up to
	   128 GB use 1 MB allocation blocks, larger disks use larger
	   allocation blocks (ie. 160 GB disk uses 0xa00 sectors 
	   allocation block (1.5 MB)).
	*/
	tf->fatitems = ((DWORD)(tf->lba_sectors) / (tf->cluster_size >> 9)) - 1;
	if (tf->fatitems > FATSIZE) {
		tf->fatitems = FATSIZE;
	}

	if (read_fat())
		return 1;
	return 0;
}

int dump_file(DWORD file)
{
	DWORD cluster = tf->dir[file].start_block;
	DWORD max_cluster = tf->dir[file].count_of_blocks;
	DWORD size = tf->cluster_size;
	FILE *fp;
	int percent = -1;
	long long clusters = 0;
	char newname[512];

	make_name(newname, tf->dir[file].name, "");
	fp = fopen(newname, "wb");
	if (fp == NULL) {
		sprintf(tf_error, "Cannot open '%s' for output\n", tf->dir[file].name);
		return 1;
	}

	do {
		if (tf_read_cluster(cluster)) return 1;
		cluster = tf->fat[cluster];
		if (cluster == 0xfffffe) {
			size = tf->cluster_size - tf->dir[file].empty_in_last_block;
		}
		if (fwrite(cluster_buffer, 1, size, fp) < size) {
			sprintf(tf_error, "Write failed (disk full?)\n");
			fclose(fp);
			return 1;
		}
		clusters++;
		if (percent != (int)(((clusters * 100LL) / max_cluster))) {
			percent = (int)((clusters * 100LL) / max_cluster);
			(tf->cb)(percent);
		}
	} while (cluster != 0xfffffe);
	fclose(fp);
	sprintf(tf_error, "\rProcessing completed.\n");
	(tf->cb)(-1);
	return 0;
}


buf_rec_file_t *open_recording_file(DWORD file, void (*cb)(int))
{
	DWORD cluster = tf->dir[file].start_block;
	buf_rec_file_t *p = malloc(sizeof(buf_rec_file_t));
	int i;

	if (!p) {
		sprintf(tf_error, "malloc for bufferred reading failed\n");
		return 0;
	}
	p->file = file;
	p->cb = cb;
	if (tf_read_cluster(cluster)) return 0;
	memcpy(p->buffer, cluster_buffer, tf->cluster_size);
	p->cluster = tf->fat[cluster];

	memcpy((void *)&(p->header), p->buffer, sizeof(recording_header_t));

	for (i = sizeof(recording_header_t); i < sizeof(recording_header_t)+sizeof(bookmarks_t)+sizeof(ts_packet); i++) {
		if (p->buffer[i] == TS_SYNC && p->buffer[i+sizeof(ts_packet)] == TS_SYNC) {
			break;
		}
	}
	if (i==sizeof(recording_header_t)+sizeof(bookmarks_t)+sizeof(ts_packet)) {
		sprintf(tf_error, "Start of TS packets not found in file %s.\n", tf->dir[file].name);
		free(p);
		return 0;
	}
	p->ptr = i; /* where packets start */
	return p;
}

void close_recording_file(buf_rec_file_t *p)
{
	if (!p) return;
	free(p);
}
	
ts_packet *read_ts_packet(buf_rec_file_t *p)
{
	ts_packet *result;
	static char local_ts_packet[sizeof(ts_packet)];
	DWORD limit;

again:
	limit = (p->cluster == 0xfffffe) ? tf->cluster_size - tf->dir[p->file].empty_in_last_block : tf->cluster_size;
	if (!limit) return 0; /* EOF */
	result = (ts_packet *)(p->buffer + p->ptr);
	if (p->ptr == limit) {
		p->ptr = sizeof(ts_packet);
		if (p->cluster != 0xfffffe) {
			if (tf_read_cluster(p->cluster)) return 0;
			memcpy(p->buffer, cluster_buffer, tf->cluster_size);
			p->cluster = tf->fat[p->cluster];
			if (p->buffer[0] != TS_SYNC) {
				p->ptr = 1;
				goto again;
			}
			return (ts_packet *)(p->buffer);
		}
		return 0; /* end of file */
	}
	if (p->ptr + sizeof(ts_packet) <= limit) {
		if (p->buffer[p->ptr] != TS_SYNC) {
			p->ptr++;
			goto again;
		}
		p->ptr += sizeof(ts_packet);
		return result;
	}
	if (p->buffer[p->ptr] != TS_SYNC) {
		p->ptr++;
		goto again;
	}
	memcpy(local_ts_packet, (void*)result, limit-p->ptr);
	if (p->cluster != 0xfffffe) {
		if (tf_read_cluster(p->cluster)) return 0;
		memcpy(p->buffer, cluster_buffer, tf->cluster_size);
		p->cluster = tf->fat[p->cluster];
		p->ptr = sizeof(ts_packet) - (limit - p->ptr);
		memcpy(local_ts_packet + sizeof(ts_packet)-p->ptr, p->buffer, p->ptr);
		return (ts_packet*)(&local_ts_packet);
	}
	return 0; /* end of file */
}

int copy_disk(void)
{
	FILE *fp;
	long long cluster = 0;
	long long max_cluster;

	int percent = -1;
	int tmp;

	max_cluster = (DWORD)(tf->lba_sectors >> 11);
	fp = fopen("tfdisk.img", "wb");
	if (fp == NULL) {
		sprintf(tf_error, "Cannot open output\n");
		return 1;
	}
	do {
		if (tf_read((__off64_t)(cluster)*tf->cluster_size, cluster_buffer, tf->cluster_size, 0)) {
			sprintf(tf_error, "Error while reading TF disk\n");
			fclose(fp);
			return 1;
		}
		if (fwrite(cluster_buffer, 1, tf->cluster_size, fp) < tf->cluster_size) {
			sprintf(tf_error, "Write failed (disk full?)\n");
			fclose(fp);
			return 1;
		}
		cluster++;
		tmp = (cluster * 100) / max_cluster;
		if (percent != tmp) {
			percent = tmp;
			(tf->cb)(percent);
		}
	} while (cluster < max_cluster);
	fclose(fp);
	sprintf(tf_error, "\nProcessing completed.\n");
	(tf->cb)(-1);
	return 0;
}

void make_name(char *newname, const char *oldname, const char *extension)
{
	strcpy(newname, oldname);
	strcat(newname, extension);
	while (*newname) {
		if (!strchr("0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ.-+=;$!@#$%^&()[]{}_", *newname))
			*newname = '_';
		newname++;
	}
}

int tf_read(__off64_t position, LPBYTE buffer, DWORD size, int swapit)
{
	DWORD bytes_read;
	int i;
	WORD *p;

	LARGE_INTEGER pos;
	pos.QuadPart = position;
	
	if (tf->lba_sectors && ((position >> 9) >= tf->lba_sectors)) {
		sprintf(tf_error, "Attempt to read after end of disk!\n");
		return 1;
	}
	if (SetFilePointer(tf->handle, pos.LowPart, &pos.HighPart, FILE_BEGIN) == INVALID_SET_FILE_POINTER) {
		sprintf(tf_error, "SetFilePointer failed on position %I64d with code %d\n", position, GetLastError());
		return 1;
	}
	if (!ReadFile(tf->handle, buffer, size, &bytes_read, NULL)) {
		sprintf(tf_error, "ReadFile failed on position %I64d with code %d\n", position, GetLastError());
		return 1;
	}

	if (bytes_read != size) {
		sprintf(tf_error, "ReadFile did not read %ld bytes but %ld bytes at position %I64d (cluster %I64d)\n", size, bytes_read, position, position/tf->cluster_size);
		return 1;
	}
	if (!swapit) return 0;
	for (i=0, p=(WORD*)buffer; i < size; i += 2, p++) {
		*p = ntohs(*p);
	}
	return 0;
}

